﻿using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using System.Windows.Forms;

namespace Xi410Ethernet
{
    public partial class Xi410EthernetForm : Form
    {
        #region Constants
        const int DEFAULT_PORT = 50101;//The Port the Camera is sending to
        const int HEIGHT = 384;//the Width of the image
        const int WIDTH = 242;//the Height of the image
        const int IMG_WIDTH = 240;//there are 242 total lines, but only 240 of them contains image information
        const int UDP_PACKAGE_LENGTH = 770;//the length of one udp package
        const int META_DATA_INDEX = 240;//Each UDP package includes a line index, the 240 line includes the meta data  
        const int LAST_EMPTY_ROW_INDEX = 241;//Each UDP package includes a line index, the 241 line is empty and shows that a full frame arived 
        const int FLAG_INDEX_IN_METADATA = 11;//the 11th byte in the metadata contains information of the flag. 
        const int IS_TEMPERATURE_MODE_IN_METADATA = 33;
        const int IS_TEMPERATURE_MODE_BIT = 3;
        #endregion

        #region Private Member
        private bool keepReading = false;//the start methds sets this to true and continue to read images from the device until 
        private UdpClient client = null;
        private int framecounter = 0;//counts the number of frame that arived. Used To calculate the frames per second.
        private DateTime startPlayTime = DateTime.Now;//The Time the recieving of Images starts. Used to calculate the frames per second.
        #endregion

        #region UI
        /// <summary>
        /// The standart WinFormContructor that initializes all UI Components
        /// </summary>
        public Xi410EthernetForm()
        {
            InitializeComponent();
            PortLabel.Text = "Port " + DEFAULT_PORT;
        }

        /// <summary>
        /// This gets called when the Form is shown and fully operational
        /// </summary>
        private void Xi410EthernetForm_Shown(object sender, EventArgs e)
        {
            Start();
        }

        /// <summary>
        /// This Gets called when the form closes
        /// </summary>
        private void Xi410EthernetForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            //When the Form closes, no more pictures should be displayed
            Stop();
            if (client != null)
            {
                client.Close();
                client = null;
            }
        }

        /// <summary>
        /// This methods shows a given Bitmap in the picturebox.
        /// This is 
        /// </summary>
        /// <param name="bitmap"></param>
        private void video_NewFrame(Bitmap bitmap, double temperature)
        {
            //if a backgroundthread wants to access an UI element it need to invoke the uithread (asks the uithread to do it for him)
            if (this.InvokeRequired)
            {
                try
                {
                    this.Invoke(new Action(() => video_NewFrame(bitmap, temperature)));
                }
                catch (ObjectDisposedException)
                {
                    //This can occur when the the form is closed but the video is not stopped. 
                }
            }
            else
            {
                pictureBox1.BackgroundImage = bitmap; // this shows the imaage in the picture box.
                frameCountLabel.Text = "FPS: " + (framecounter / (DateTime.Now - startPlayTime).TotalSeconds).ToString("0.0"); // 
                framecounter++;
                CurrentTemperatureLabel.Text = "Current Temperature: " + temperature.ToString("0.0") + "°C";
            }
        }

        #endregion

        #region Image Aquisition
        private void Start()
        {
            Stop();//Stop image aquisition when one is already active
            framecounter = 0;//reset the framecounter. Needed to calculate the frames per seconds
            startPlayTime = DateTime.Now;//remember the starttime. Needed to calculate the frames per seconds 

            //Start a backgroundThread which constantly checks the udp - port for new packages, collects them in an array
            //then converts them to a bitmap and sends it to a picturebox.
            Thread t = new Thread(Run);
            t.Start();
        }

        /// <summary>
        /// this stop the while loop in Run. The current processed Image will be finished, but no new one will be started
        /// </summary>
        private void Stop()
        {
            keepReading = false;

        }
        /// <summary>
        /// The opens a UDP-Client and in a loop recieves and processes the incoming data
        /// </summary>
        private void Run()
        {
            try
            {
                int port = DEFAULT_PORT;
                IPEndPoint remoteEP = new IPEndPoint(IPAddress.Any, port);//listen to any IP Address on the port
                client = new UdpClient(port);//Create a new UdpClient
                double[] newImg = new double[WIDTH * HEIGHT];//Initiate an array that stores the temperature of a frame 
                double[] lastImg = new double[WIDTH * HEIGHT];//If the flag is close, the new img shows wrong temperatures,  so wie show the last valid one        
                int lastValidFrameCounter = 0;//remember the last valid framecounter
                byte[] metaData = new byte[UDP_PACKAGE_LENGTH - 2];//Initiate an array that hold the meta information of a frame, the first 2 bytes of the package indicate the line and frame and do not belong to the metadata
                keepReading = true; //keep reading 
                while (keepReading)//Continue reading the the udp client until keepReading is set to false, i.e. the programm is closed or the stop button is clicked
                {
                    byte[] data = client.Receive(ref remoteEP);//Wait for the next udp package to arive. When the firewall blockes this application, this takes forever
                    if (data.Length != UDP_PACKAGE_LENGTH) continue;//check that the package has length 770
                    int offset = data[0] * HEIGHT;//Each package has a row counter on byte 0 , a framecounter on byte 1 and the 2*384 bytes for the temperatures of the pixels
                    for (int i = 0; i < HEIGHT; i++)//iterate over the width
                    {
                        newImg[i + offset] = (data[i * 2 + 3] * 256 + data[i * 2 + 2] - 1000) / 10d; // use the temperature format t = (byte1*256+byte2 -1000)/10
                    }
                    if (data[0] == META_DATA_INDEX) Array.Copy(data, 2, metaData, 0, 768); // MetaData comes in the 241th row
                    if (data[0] == LAST_EMPTY_ROW_INDEX)//this page is empty and tells us a new img has arived
                    {

                        //The Temperature Flag indicates that the image contains temperature and not adu values
                        //If not set, asks the user to set it via Pix Connect
                        if (!IsTemperatureFlag(metaData))
                        {
                            keepReading = false;
                            MessageBox.Show("The Xi410 is set in ADU mode and not in Temperature mode. Please go the the PixConnect and change that.");
                        }
                        byte flagstate = metaData[FLAG_INDEX_IN_METADATA];//get the flag status
                        if (flagstate == 0 && newImg[HEIGHT * HEIGHT / 2 + WIDTH / 2] < 100)//take the image only if the new tempeture is valid
                        {

                            Array.Copy(newImg, lastImg, WIDTH * HEIGHT);//save the image as last image
                            lastValidFrameCounter = data[1];//remeber the last valid framecounter
                        }

                        ReadyNewFrame(lastImg);
                    }
                }
                client.Close();
                client = null;
            }
            //If the Connection breaks
            catch (SocketException ex)
            {
                if (client != null)
                {
                    client.Close();
                    client = null;
                }
                System.Diagnostics.Debug.WriteLine(ex.ToString());
            }
        }

        private static bool IsTemperatureFlag(byte[] metaData)
        {
            //First Chech if MetaData are valid
            if (metaData == null || metaData.Length < UDP_PACKAGE_LENGTH - 2)
            {
                return false;
            }
            //Check if the 3rd bit in the metadatas 33th byte is set
            return ((metaData[IS_TEMPERATURE_MODE_IN_METADATA]) & (1 << IS_TEMPERATURE_MODE_BIT)) > 0;
        }

        /// <summary>
        /// this methods transforms the incoming temperature data to a Bitmap and then calls video_NewFrame to show it in the picturebox
        /// </summary>
        /// <param name="img">the whole frame with temperatures on </param>
        private void ReadyNewFrame(double[] img)
        {
            //Get 3 sigma min and max value of the image;
            int img_length = IMG_WIDTH * HEIGHT;

            //Calculate the mean value
            double sum = 0;
            for (int i = 0; i < img_length; i++)
            {
                sum += img[i];
            }

            double mean = (double)sum / img_length;

            //Calculate the Variance
            sum = 0;
            for (int i = 0; i < img_length; i++)
            {
                sum += (mean - img[i]) * (mean - img[i]);
            }
            double variance = sum / img_length;
            variance = Math.Sqrt(variance);
            variance *= 3;  // 3 Sigma

            //Calculate min and max
            double min = mean - variance;
            double max = mean + variance;

            //Create a two-dimensional array with gray values
            double[,] rawImage = new double[IMG_WIDTH, HEIGHT];
            for (int y = 0; y < rawImage.GetLength(1); y++)
            {
                for (int x = 0; x < rawImage.GetLength(0); x++)
                {
                    double d = img[x * HEIGHT + y];
                    if (d > max)
                    {
                        rawImage[x, y] = 1;
                    }
                    else if (d < min)
                    {
                        rawImage[x, y] = 0;
                    }
                    else
                    {
                        rawImage[x, y] = (d - min) / (max - min);

                    }
                }
            }
            //Convert gray image to bitmap
            Bitmap bitmap = ToBitmap(rawImage);

            //Lauch New Frame Event with bitmap and metadatas of the frame
            video_NewFrame(bitmap, img[img.Length / 2]);
        }

        /// <summary>
        /// Transforms a double array of Gray data, i.e. values between 0 and 1, into a bitmap.
        /// </summary>
        /// <param name="rawImage">The Raw  gray image</param>
        /// <returns>The Bitmap of the gray image</returns>
        private static unsafe Bitmap ToBitmap(double[,] rawImage)
        {
            int width = rawImage.GetLength(1);
            int height = rawImage.GetLength(0);

            Bitmap image = new Bitmap(width, height);
            BitmapData bitmapData = image.LockBits(
                new Rectangle(0, 0, width, height),
                ImageLockMode.ReadWrite,
                PixelFormat.Format32bppArgb
            );
            ColorARGB* startingPosition = (ColorARGB*)bitmapData.Scan0;


            for (int i = 0; i < height; i++)
            {
                for (int j = 0; j < width; j++)
                {
                    double color = rawImage[i, j];
                    byte rgb = (byte)(color * 255);

                    ColorARGB* position = startingPosition + j + i * width;
                    position->A = 255;
                    position->R = rgb;
                    position->G = rgb;
                    position->B = rgb;
                }
            }

            image.UnlockBits(bitmapData);
            return image;
        }

        /// <summary>
        /// This defines a structure of 32ARGB Color, one byte each for alpha, red, green, red
        /// A (from 0 to 255) represents alpha (translucency) 
        /// R (from 0 to 255) represents red 
        /// G (from 0 to 255) represents green 
        /// B (from 0 to 255) represents blue 
        /// </summary>
        public struct ColorARGB
        {
            public byte B;
            public byte G;
            public byte R;
            public byte A;

            public ColorARGB(Color color)
            {
                A = color.A;
                R = color.R;
                G = color.G;
                B = color.B;
            }

            public ColorARGB(byte a, byte r, byte g, byte b)
            {
                A = a;
                R = r;
                G = g;
                B = b;
            }

            public Color ToColor()
            {
                return Color.FromArgb(A, R, G, B);
            }
        }
        #endregion

    }
}